class ChatWidget {
    constructor(options = {}) {
        this.options = {
            apiUrl: options.apiUrl || '/api/',
            wsUrl: options.wsUrl || 'ws://localhost:8080',
            userCode: options.userCode,
            position: options.position || 'bottom-right',
            theme: options.theme || 'light',
            ...options
        };

        this.init();
    }

    init() {
        this.createWidget();
        this.loadUserData();
        this.setupEventListeners();
    }

    createWidget() {
        // Создание HTML структуры виджета
        this.widget = document.createElement('div');
        this.widget.className = `chat-widget chat-widget-${this.options.theme} chat-widget-${this.options.position}`;
        this.widget.innerHTML = `
            <div class="chat-widget-header">
                <span>Чат поддержки</span>
                <button class="chat-widget-toggle">−</button>
            </div>
            <div class="chat-widget-body">
                <div class="chat-messages"></div>
                <div class="chat-input-container">
                    <textarea placeholder="Введите сообщение..."></textarea>
                    <button class="send-button">Отправить</button>
                </div>
            </div>
        `;

        document.body.appendChild(this.widget);
    }

    async loadUserData() {
        try {
            const response = await fetch(`${this.options.apiUrl}auth.php`, {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({user_code: this.options.userCode})
            });

            const data = await response.json();
            
            if (data.success) {
                this.user = data.user;
                this.connectWebSocket();
                this.loadChatHistory();
            }
        } catch (error) {
            console.error('Ошибка загрузки данных пользователя:', error);
        }
    }

    connectWebSocket() {
        this.ws = new WebSocket(this.options.wsUrl);
        
        this.ws.onopen = () => {
            this.ws.send(JSON.stringify({
                action: 'auth',
                user_id: this.user.id
            }));
        };

        this.ws.onmessage = (event) => {
            const data = JSON.parse(event.data);
            this.handleIncomingMessage(data);
        };
    }

    handleIncomingMessage(message) {
        this.addMessageToChat(message, false);
    }

    addMessageToChat(message, isOwn = true) {
        const messagesContainer = this.widget.querySelector('.chat-messages');
        const messageElement = document.createElement('div');
        messageElement.className = `message ${isOwn ? 'own-message' : 'other-message'}`;
        messageElement.innerHTML = `
            <div class="message-content">${message.message_text}</div>
            <div class="message-time">${new Date().toLocaleTimeString()}</div>
        `;
        
        messagesContainer.appendChild(messageElement);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }

    setupEventListeners() {
        const sendButton = this.widget.querySelector('.send-button');
        const textarea = this.widget.querySelector('textarea');
        const toggleButton = this.widget.querySelector('.chat-widget-toggle');

        sendButton.addEventListener('click', () => this.sendMessage());
        textarea.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.sendMessage();
            }
        });

        toggleButton.addEventListener('click', () => {
            this.widget.classList.toggle('minimized');
        });
    }

    async sendMessage() {
        const textarea = this.widget.querySelector('textarea');
        const message = textarea.value.trim();

        if (!message) return;

        try {
            const response = await fetch(`${this.options.apiUrl}messages.php`, {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    user_id: this.user.id,
                    message: message,
                    chat_type: 'support'
                })
            });

            if (response.ok) {
                this.addMessageToChat({message_text: message}, true);
                textarea.value = '';
            }
        } catch (error) {
            console.error('Ошибка отправки сообщения:', error);
        }
    }
}

// Использование виджета
// new ChatWidget({
//     userCode: 'UNIQUE_USER_CODE',
//     position: 'bottom-right',
//     theme: 'dark'
// });