<?php
use Ratchet\MessageComponentInterface;
use Ratchet\ConnectionInterface;
use Ratchet\Server\IoServer;
use Ratchet\Http\HttpServer;
use Ratchet\WebSocket\WsServer;

require __DIR__ . '/../vendor/autoload.php';

class ChatWebSocket implements MessageComponentInterface {
    protected $clients;
    protected $users;

    public function __construct() {
        $this->clients = new \SplObjectStorage;
        $this->users = [];
    }

    public function onOpen(ConnectionInterface $conn) {
        $this->clients->attach($conn);
        echo "New connection! ({$conn->resourceId})\n";
    }

    public function onMessage(ConnectionInterface $from, $msg) {
        $data = json_decode($msg, true);
        
        if (!$data) return;
        
        switch ($data['action']) {
            case 'auth':
                $this->users[$from->resourceId] = $data['user_id'];
                echo "User {$data['user_id']} connected\n";
                break;
                
            case 'message':
                $this->sendMessageToRecipients($data);
                break;
                
            case 'typing':
                $this->broadcastTypingStatus($data);
                break;
        }
    }

    public function onClose(ConnectionInterface $conn) {
        if (isset($this->users[$conn->resourceId])) {
            $userId = $this->users[$conn->resourceId];
            echo "User {$userId} disconnected\n";
            unset($this->users[$conn->resourceId]);
        }
        $this->clients->detach($conn);
    }

    public function onError(ConnectionInterface $conn, \Exception $e) {
        echo "Error: {$e->getMessage()}\n";
        $conn->close();
    }

    private function sendMessageToRecipients($data) {
        foreach ($this->clients as $client) {
            if (in_array($this->users[$client->resourceId], $data['recipients'])) {
                $client->send(json_encode([
                    'type' => 'message',
                    'data' => $data
                ]));
            }
        }
        
        // Сохраняем сообщение в БД
        $this->saveMessageToDB($data);
    }

    private function broadcastTypingStatus($data) {
        foreach ($this->clients as $client) {
            if (in_array($this->users[$client->resourceId], $data['recipients'])) {
                $client->send(json_encode([
                    'type' => 'typing',
                    'data' => $data
                ]));
            }
        }
    }

    private function saveMessageToDB($data) {
        // Здесь код для сохранения сообщения в базу данных
    }
}

// Запуск сервера
$server = IoServer::factory(
    new HttpServer(
        new WsServer(
            new ChatWebSocket()
        )
    ),
    8080
);

$server->run();
?>