class ChatWebSocket {
    constructor() {
        this.ws = null;
        this.connected = false;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 5;
        
        this.connect();
    }

    connect() {
        const protocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
        this.ws = new WebSocket(`${protocol}//${window.location.hostname}:8080`);

        this.ws.onopen = () => {
            console.log('WebSocket connected');
            this.connected = true;
            this.reconnectAttempts = 0;
            
            // Авторизация
            this.send({
                action: 'auth',
                user_id: window.userId
            });
        };

        this.ws.onmessage = (event) => {
            const data = JSON.parse(event.data);
            this.handleMessage(data);
        };

        this.ws.onclose = () => {
            console.log('WebSocket disconnected');
            this.connected = false;
            this.handleReconnect();
        };

        this.ws.onerror = (error) => {
            console.error('WebSocket error:', error);
        };
    }

    send(data) {
        if (this.connected) {
            this.ws.send(JSON.stringify(data));
        }
    }

    handleMessage(data) {
        switch (data.type) {
            case 'message':
                this.onMessageReceived(data.data);
                break;
            case 'typing':
                this.onTypingStatus(data.data);
                break;
        }
    }

    onMessageReceived(message) {
        // Обработка нового сообщения
        if (window.currentChat && 
            ((message.chat_id && message.chat_id === window.currentChat.id) || 
             (message.group_id && message.group_id === window.currentChat.id))) {
            this.addMessageToChat(message);
        }
    }

    onTypingStatus(data) {
        // Показать статус "печатает"
    }

    handleReconnect() {
        if (this.reconnectAttempts < this.maxReconnectAttempts) {
            this.reconnectAttempts++;
            setTimeout(() => this.connect(), 3000);
        }
    }

    addMessageToChat(message) {
        const messagesContainer = document.getElementById('messagesContainer');
        const messageElement = this.createMessageElement(message);
        messagesContainer.appendChild(messageElement);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }

    createMessageElement(message) {
        // Создание элемента сообщения
        const div = document.createElement('div');
        div.className = `message ${message.user_id === window.userId ? 'own-message' : 'other-message'}`;
        div.innerHTML = `
            <div class="message-content">
                ${message.message_text ? `<p>${this.escapeHtml(message.message_text)}</p>` : ''}
                ${message.file_path ? `<div class="file-attachment">Файл: ${message.file_name}</div>` : ''}
            </div>
            <div class="message-time">${this.formatTime(message.created_at)}</div>
        `;
        return div;
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    formatTime(timestamp) {
        return new Date(timestamp).toLocaleTimeString('ru-RU', {
            hour: '2-digit',
            minute: '2-digit'
        });
    }
}

// Инициализация WebSocket при загрузке страницы
document.addEventListener('DOMContentLoaded', function() {
    window.chatWebSocket = new ChatWebSocket();
});