<?php
// index.php
require_once 'config/config.php';
require_once 'config/database.php';
require_once 'classes/Chat.php';
require_once 'classes/Message.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$database = new Database();
$chat = new Chat($database);
$message = new Message($database);

$userChats = $chat->getUserChats($_SESSION['user_id']);
$currentChatId = $_GET['chat'] ?? null;
$messages = $currentChatId ? $message->getMessages($currentChatId) : [];
?>

<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Чат-приложение</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .chat-container { height: 100vh; }
        .chat-sidebar { height: 100vh; overflow-y: auto; border-right: 1px solid #dee2e6; }
        .messages-container { height: calc(100vh - 120px); overflow-y: auto; }
        .message-input { position: fixed; bottom: 0; right: 0; left: 250px; }
        .message-bubble { max-width: 70%; margin: 5px 0; }
        .message-bubble.own { margin-left: auto; background-color: #007bff; color: white; }
    </style>
</head>
<body>
    <div class="container-fluid chat-container">
        <div class="row h-100">
            <!-- Боковая панель с чатами -->
            <div class="col-md-3 chat-sidebar p-3">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h5>Чаты</h5>
                    <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#joinChatModal">
                        Присоединиться
                    </button>
                </div>
                
                <!-- Поиск -->
                <input type="text" class="form-control mb-3" id="searchInput" placeholder="Поиск сообщений...">
                
                <!-- Список чатов -->
                <div class="chat-list">
                    <?php foreach ($userChats as $chatItem): ?>
                        <div class="chat-item p-2 border rounded mb-2 <?= $currentChatId == $chatItem['id'] ? 'bg-primary text-white' : '' ?>"
                             onclick="selectChat(<?= $chatItem['id'] ?>)" style="cursor: pointer;">
                            <div class="fw-bold"><?= htmlspecialchars($chatItem['name']) ?></div>
                            <small><?= $chatItem['type'] === 'group' ? 'Группа' : 'Личный чат' ?></small>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- Область сообщений -->
            <div class="col-md-9 p-0">
                <?php if ($currentChatId): ?>
                    <div class="messages-container p-3" id="messagesContainer">
                        <?php foreach ($messages as $msg): ?>
                            <div class="message-bubble p-2 rounded <?= $msg['user_id'] == $_SESSION['user_id'] ? 'own' : 'bg-light' ?>">
                                <small class="text-muted"><?= htmlspecialchars($msg['username']) ?></small>
                                <div><?= htmlspecialchars($msg['message']) ?></div>
                                <small class="text-muted"><?= date('H:i', strtotime($msg['created_at'])) ?></small>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <!-- Поле ввода сообщения -->
                    <div class="message-input p-3 bg-white border-top">
                        <div class="input-group">
                            <input type="text" class="form-control" id="messageInput" placeholder="Введите сообщение...">
                            <button class="btn btn-primary" onclick="sendMessage()">Отправить</button>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="d-flex align-items-center justify-content-center h-100">
                        <h4 class="text-muted">Выберите чат для начала общения</h4>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Модальное окно присоединения к чату -->
    <div class="modal fade" id="joinChatModal">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Присоединиться к чату</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="text" class="form-control" id="chatCode" placeholder="Введите код чата">
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-primary" onclick="joinChat()">Присоединиться</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        let ws;
        const currentChatId = <?= $currentChatId ?: 'null' ?>;
        const userId = <?= $_SESSION['user_id'] ?>;
        const username = '<?= $_SESSION['username'] ?>';

        // WebSocket подключение
        if (currentChatId) {
            ws = new WebSocket('ws://localhost:8080');
            
            ws.onopen = function() {
                ws.send(JSON.stringify({
                    type: 'join',
                    chatId: currentChatId,
                    userId: userId
                }));
            };

            ws.onmessage = function(event) {
                const data = JSON.parse(event.data);
                if (data.type === 'newMessage') {
                    addMessageToChat(data);
                }
            };
        }

        function sendMessage() {
            const input = document.getElementById('messageInput');
            const message = input.value.trim();
            
            if (message && currentChatId) {
                // Отправка через AJAX
                fetch('api/send_message.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({
                        chatId: currentChatId,
                        message: message
                    })
                }).then(() => {
                    // Отправка через WebSocket
                    ws.send(JSON.stringify({
                        type: 'message',
                        chatId: currentChatId,
                        message: message,
                        username: username
                    }));
                    input.value = '';
                });
            }
        }

        function addMessageToChat(data) {
            const container = document.getElementById('messagesContainer');
            const messageDiv = document.createElement('div');
            messageDiv.className = 'message-bubble p-2 rounded bg-light';
            messageDiv.innerHTML = `
                <small class="text-muted">${data.username}</small>
                <div>${data.message}</div>
                <small class="text-muted">${data.timestamp}</small>
            `;
            container.appendChild(messageDiv);
            container.scrollTop = container.scrollHeight;
        }

        function selectChat(chatId) {
            window.location.href = `?chat=${chatId}`;
        }

        function joinChat() {
            const code = document.getElementById('chatCode').value;
            if (code) {
                fetch('api/join_chat.php', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({code: code})
                }).then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Чат не найден или ошибка при присоединении');
                    }
                });
            }
        }

        // Поиск сообщений
        document.getElementById('searchInput').addEventListener('input', function() {
            const query = this.value;
            if (query.length > 2) {
                fetch(`api/search.php?q=${encodeURIComponent(query)}`)
                    .then(response => response.json())
                    .then(data => {
                        // Отобразить результаты поиска
                        console.log(data);
                    });
            }
        });

        // Отправка сообщения по Enter
        document.getElementById('messageInput')?.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                sendMessage();
            }
        });
    </script>
</body>
</html>