<?php
// classes/Chat.php
class Chat {
    private $db;

    public function __construct($database) {
        $this->db = $database->connect();
    }

    public function createChat($name, $type, $description, $createdBy) {
        $chatCode = $this->generateChatCode();
        $sql = "INSERT INTO chats (name, type, chat_code, description, created_by) 
                VALUES (?, ?, ?, ?, ?)";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$name, $type, $chatCode, $description, $createdBy]);
        
        $chatId = $this->db->lastInsertId();
        
        // Добавляем создателя как администратора
        $this->addMember($chatId, $createdBy, 'admin');
        
        return ['id' => $chatId, 'code' => $chatCode];
    }

    public function joinChatByCode($chatCode, $userId) {
        $sql = "SELECT id FROM chats WHERE chat_code = ?";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$chatCode]);
        $chat = $stmt->fetch();
        
        if ($chat) {
            return $this->addMember($chat['id'], $userId);
        }
        return false;
    }

    public function addMember($chatId, $userId, $role = 'member') {
        $sql = "INSERT IGNORE INTO chat_members (chat_id, user_id, role) VALUES (?, ?, ?)";
        $stmt = $this->db->prepare($sql);
        return $stmt->execute([$chatId, $userId, $role]);
    }

    public function getUserChats($userId) {
        $sql = "SELECT c.*, cm.role FROM chats c 
                JOIN chat_members cm ON c.id = cm.chat_id 
                WHERE cm.user_id = ? 
                ORDER BY c.created_at DESC";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }

    private function generateChatCode() {
        return strtoupper(substr(uniqid(), -8));
    }

    public function searchMessages($userId, $query) {
        $sql = "SELECT m.*, c.name as chat_name, u.username 
                FROM messages m
                JOIN chats c ON m.chat_id = c.id
                JOIN users u ON m.user_id = u.id
                JOIN chat_members cm ON c.id = cm.chat_id AND cm.user_id = ?
                WHERE MATCH(m.message) AGAINST(? IN BOOLEAN MODE)
                ORDER BY m.created_at DESC LIMIT 50";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId, $query]);
        return $stmt->fetchAll();
    }
}

// classes/Message.php
class Message {
    private $db;

    public function __construct($database) {
        $this->db = $database->connect();
    }

    public function sendMessage($chatId, $userId, $message) {
        $sql = "INSERT INTO messages (chat_id, user_id, message) VALUES (?, ?, ?)";
        $stmt = $this->db->prepare($sql);
        $result = $stmt->execute([$chatId, $userId, $message]);
        
        if ($result) {
            $messageId = $this->db->lastInsertId();
            $this->createNotifications($chatId, $userId, $messageId);
            return $messageId;
        }
        return false;
    }

    public function getMessages($chatId, $limit = 50, $offset = 0) {
        $sql = "SELECT m.*, u.username, u.avatar 
                FROM messages m
                JOIN users u ON m.user_id = u.id
                WHERE m.chat_id = ?
                ORDER BY m.created_at DESC
                LIMIT ? OFFSET ?";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$chatId, $limit, $offset]);
        return array_reverse($stmt->fetchAll());
    }

    private function createNotifications($chatId, $senderId, $messageId) {
        $sql = "INSERT INTO notifications (user_id, chat_id, message_id)
                SELECT cm.user_id, ?, ?
                FROM chat_members cm
                WHERE cm.chat_id = ? AND cm.user_id != ?";
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$chatId, $messageId, $chatId, $senderId]);
    }
}
?>