<?php
// api/upload_file.php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';
session_start();

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['error' => 'Not authenticated']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST' || !isset($_FILES['file'])) {
    echo json_encode(['error' => 'No file uploaded']);
    exit;
}

$file = $_FILES['file'];
$userId = $_SESSION['user_id'];
$chatId = $_POST['chatId'] ?? null;

// Проверка размера файла (макс 50MB)
if ($file['size'] > 50 * 1024 * 1024) {
    echo json_encode(['error' => 'File too large']);
    exit;
}

// Разрешенные типы файлов
$allowedMimes = [
    'image/jpeg', 'image/png', 'image/gif', 'image/webp',
    'application/pdf', 'text/plain', 'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'video/mp4', 'video/avi', 'audio/mpeg', 'audio/wav'
];

if (!in_array($file['type'], $allowedMimes)) {
    echo json_encode(['error' => 'File type not allowed']);
    exit;
}

// Определение типа файла
function getFileType($mimeType) {
    if (strpos($mimeType, 'image/') === 0) return 'image';
    if (strpos($mimeType, 'video/') === 0) return 'video';
    if (strpos($mimeType, 'audio/') === 0) return 'audio';
    if (in_array($mimeType, ['application/pdf', 'text/plain', 'application/msword', 
                            'application/vnd.openxmlformats-officedocument.wordprocessingml.document'])) {
        return 'document';
    }
    return 'other';
}

// Создание директории uploads если не существует
$uploadDir = '../uploads/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Генерация уникального имени файла
$fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);
$storedName = uniqid() . '.' . $fileExtension;
$filePath = $uploadDir . $storedName;

if (move_uploaded_file($file['tmp_name'], $filePath)) {
    // Создание миниатюры для изображений
    if (getFileType($file['type']) === 'image') {
        createThumbnail($filePath, $uploadDir . 'thumb_' . $storedName);
    }
    
    // Сохранение информации о файле в БД
    $stmt = $pdo->prepare("INSERT INTO files (original_name, stored_name, file_type, mime_type, file_size, uploaded_by) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->execute([
        $file['name'],
        $storedName,
        getFileType($file['type']),
        $file['type'],
        $file['size'],
        $userId
    ]);
    
    $fileId = $pdo->lastInsertId();
    
    // Если указан чат, отправляем сообщение с файлом
    if ($chatId) {
        $stmt = $pdo->prepare("INSERT INTO messages (chat_id, user_id, file_id, created_at) VALUES (?, ?, ?, NOW())");
        $stmt->execute([$chatId, $userId, $fileId]);
        
        $messageId = $pdo->lastInsertId();
        
        // Получение информации для WebSocket
        $stmt = $pdo->prepare("
            SELECT m.id, m.content, m.created_at, u.username, u.avatar,
                   f.id as file_id, f.original_name, f.stored_name, f.file_type, f.mime_type, f.file_size
            FROM messages m 
            JOIN users u ON m.user_id = u.id 
            LEFT JOIN files f ON m.file_id = f.id
            WHERE m.id = ?
        ");
        $stmt->execute([$messageId]);
        $messageData = $stmt->fetch();
        
        // Отправка через WebSocket
        $socketData = [
            'type' => 'new_message',
            'chatId' => $chatId,
            'message' => $messageData
        ];
        
        sendWebSocketMessage($socketData);
    }
    
    echo json_encode([
        'success' => true,
        'fileId' => $fileId,
        'fileName' => $file['name'],
        'fileType' => getFileType($file['type']),
        'fileSize' => $file['size']
    ]);
} else {
    echo json_encode(['error' => 'Failed to upload file']);
}

function createThumbnail($source, $destination, $maxWidth = 200, $maxHeight = 200) {
    $imageInfo = getimagesize($source);
    if (!$imageInfo) return false;
    
    $width = $imageInfo[0];
    $height = $imageInfo[1];
    $mimeType = $imageInfo['mime'];
    
    // Расчет новых размеров
    $ratio = min($maxWidth / $width, $maxHeight / $height);
    $newWidth = intval($width * $ratio);
    $newHeight = intval($height * $ratio);
    
    // Создание изображения из источника
    switch ($mimeType) {
        case 'image/jpeg':
            $sourceImage = imagecreatefromjpeg($source);
            break;
        case 'image/png':
            $sourceImage = imagecreatefrompng($source);
            break;
        case 'image/gif':
            $sourceImage = imagecreatefromgif($source);
            break;
        default:
            return false;
    }
    
    // Создание миниатюры
    $thumbnail = imagecreatetruecolor($newWidth, $newHeight);
    imagecopyresampled($thumbnail, $sourceImage, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
    
    // Сохранение миниатюры
    imagejpeg($thumbnail, $destination, 85);
    
    imagedestroy($sourceImage);
    imagedestroy($thumbnail);
    
    return true;
}
?>